<?php

namespace App\Http\Controllers;

use App\Models\Appointment;
use App\Models\Doctor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class AppointmentController extends Controller
{
    public function create()
    {
        $doctors = Doctor::where('is_active', true)->get();
        
        return view('appointment', compact('doctors'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'age' => 'required|integer|min:1|max:150',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'preferred_time' => 'required|date|after:now',
            'doctor_id' => 'nullable|exists:doctors,id',
            'message' => 'nullable|string|max:1000',
        ]);

        $appointment = Appointment::create($validated);

        // Send email notification
        try {
            Mail::send('emails.appointment-notification', ['appointment' => $appointment], function ($message) {
                $message->to(config('mail.from.address'))
                    ->subject('New Appointment Request - ENT Care Center');
            });
        } catch (\Exception $e) {
            // Log error but don't fail the request
            \Log::error('Failed to send appointment email: ' . $e->getMessage());
        }

        return redirect()->route('appointment.create')
            ->with('success', 'Your appointment request has been submitted successfully! We will contact you soon.');
    }
}
